/******************************************************************************
 * mach64 Chapter 5 sample code                                               *
 *                                                                            *
 * This program performs mach64 detection, aperture setting, and mode         *
 * setting.  It then sets up the engine.  After that, mode information        *
 * is displayed.                                                              *
 *                                                                            *
 * Copyright (c) 1994-1998 ATI Technologies Inc.  All rights reserved.        *
 ******************************************************************************/

#include <stdio.h>
#include <i86.h>
#include "atim64.h"
#include "defines.h"
#include "main.h"

/******************************************************************************
 * Main Program                                                               *
 *  Function: Demonstrates mach64 engine initialization                       *
 *    Inputs: NONE                                                            *
 *   Outputs: NONE                                                            *
 ******************************************************************************/

void main (int argc, char *argv[])
{
    printf ("mach64 Chapter 5 sample code\n"
            "\n"
            "engine.c\n"
            "This program performs mach64 detection, aperture setting, and mode\n"
            "setting.  It then sets up the mach64 engine.  After, the query data\n"
            "and mode information are displayed.\n"
            "\n"
            "Spatial resolution (640, 800, 1024, 1280, 1600) and Colour Depth\n"
            "(8, 15, 16, 24, 32) should be passed as arguments.\n"
            "Default setting is 640x480 spatial resolution and 8bpp pixel depth.\n");

    // Detect mach64.
    if (!init_graphics ())
    {
        printf ("ERROR: Graphics not initialized properly!\n");
        exit (0);
    } // if

    // Fill QUERY_DATA using BIOS calls 08h and 09h and set globals.
    if (!long_query ())
    {
        printf ("ERROR: long query failed!\n");
        exit (0);
    } // if

    // Save old video mode.
    get_old_mode ();

    // Process the command line arguments to override default resolution
    // and colour depth settings.
    process_command_line (argc, argv);
    printf ("Spatial X-Resolution: %d\n", GMODE_RES);
    printf ("         Pixel Depth: %d\n\n", GCLR_DEPTH);
    printf (" Press Any Key To Begin\n");
    getch ();

    // Load and set mode and set globals.  '2' indicates pitch is same as xres.
    if (!load_and_set_mode (GMODE_RES, GCLR_DEPTH, 2))
    {
        printf ("ERROR: mode setting failed!\n");
        exit (0);
    } // if

    // Enable aperture.
    if (!enable_aperture ())
    {
        printf ("ERROR: enable aperture failed!\n");
        exit (0);
    } // if

    // Setup engine context.
    // Please see init.c for full details of this function.
    init_engine ();

    // Sample drawing routines go here.
    getch ();

    // Ensure DAC is set to 6 bit operation if chip is a CT.
    if (ior16 (CFG_CHIP_TYPE) == CHIP_CT_ID)
    {
        iow16 (DAC_CNTL, ior16 (DAC_CNTL) & 0xFEFF);
    } // if

    // Restore the default palette if it was modified.
    if ((MODE_INFO.bpp == 4) || (MODE_INFO.bpp == 8) ||
        (ior16 (CFG_CHIP_TYPE) == CHIP_CT_ID))
    {
        restore_palette (SAVEPALETTE);
    } // if

    // Restore mode.
    set_old_mode ();
    MODE_INFO.vga_aperture_status = OFF;

    printf ("QUERY_DATA information\n");
    printf (" size                  %04x\n", QUERY_DATA.size);
    printf ("revision               %02x\n", QUERY_DATA.revision);
    printf ("mode_tables            %02x\n", QUERY_DATA.mode_tables);
    printf (" mode_table_offset     %04x\n", QUERY_DATA.mode_table_offset);
    printf ("mode_table_size        %02x\n", QUERY_DATA.mode_table_size);
    printf ("vga_type               %02x\n", QUERY_DATA.vga_type);
    printf (" asic_id               %04x\n", QUERY_DATA.asic_id);
    printf ("vga_boundary           %02x\n", QUERY_DATA.vga_boundary);
    printf ("memory_size            %02x\n", QUERY_DATA.memory_size);
    printf ("dac_type               %02x\n", QUERY_DATA.dac_type);
    printf ("memory_type            %02x\n", QUERY_DATA.memory_type);
    printf ("bus_type               %02x\n", QUERY_DATA.bus_type);
    printf ("monitor_cntl           %02x\n", QUERY_DATA.monitor_cntl);
    printf (" aperture_addr         %04x\n", QUERY_DATA.aperture_addr);
    printf ("aperture_cfg           %02x\n", QUERY_DATA.aperture_cfg);
    printf ("colour_depth_support   %02x\n", QUERY_DATA.colour_depth_support);
    printf ("ramdac_feature_support %02x\n", QUERY_DATA.ramdac_feature_support);
    printf ("io_type                %02x\n", QUERY_DATA.io_type);
    printf (" reserved1             %04x\n", QUERY_DATA.reserved1);
    printf (" io_base               %04x\n", QUERY_DATA.io_base);
    printf ("reserved2[6]           %02x\n\n", QUERY_DATA.reserved2[6]);
    getch ();

    // Display I/O data info and mode info.
    printf ("IO_DATA:\n");
    printf ("   io_base:%x\n", IO_DATA.io_base);
    printf ("   io_type:%d\n", IO_DATA.io_type);
    printf ("MODE_INFO:\n");
    printf ("   xres:%d\n", MODE_INFO.xres);
    printf ("   yres:%d\n", MODE_INFO.yres);
    printf ("   bpp:%d\n", MODE_INFO.bpp);
    printf ("   depth:%d\n", MODE_INFO.depth);
    printf ("   pitch:%d\n", MODE_INFO.pitch);
    printf ("   vga_aperture_status:%d\n", MODE_INFO.vga_aperture_status);
    printf ("   linear_aperture_status:%d\n", MODE_INFO.linear_aperture_status);
    printf ("   linear_aperture_size:%d\n", MODE_INFO.linear_aperture_size);
    printf ("   linear_address:%lx\n", MODE_INFO.linear_address);
    printf ("   virt_seg:%lx\n", MODE_INFO.virt_seg);
    printf ("   vga_memreg_offset:%lx\n", MODE_INFO.vga_memreg_offset);
    printf ("   linear_memreg_offset:%lx\n", MODE_INFO.linear_memreg_offset);
    printf ("\nPress any key to end\n");
    getch ();

    exit (0);                           // No errors.

} // main
