/*==========================================================================
  GPLINE.C

  Example code to draw a general pattern line. Both directions are shown
  (left to right, right to left).

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"

#define INSTANCE 0

/* --------------------------------------------------------------------------
  HOST_FILL - writes data to screen through host register

  This routine writes data to the screen (for a previously set rectangle
  fill). The data is packed into 32 bit pieces since each host register
  write requires 32 bits of data.
-------------------------------------------------------------------------- */
void host_fill (int src_line_length)
{
    int loop, color, colorcount, colorcycle;
    int shifts, maxshifts, remainder;
    unsigned long hostdata, pixel;
    unsigned long shifter, totalbits;
    unsigned long hostwrts;

    // Determine if there is an even number of host writes - if not, the
    // remainder must be writen to a host register outside the main loop.
    // The extra bits not used will be thrown away by the engine. Also, this
    // insures that the engine will not be busy waiting for more host data.

    remainder = 0;
    totalbits = (unsigned long) (src_line_length) *
                (unsigned long) (modeinfo.bpp);
    hostwrts = totalbits / 32;
    if ((hostwrts * 32) != totalbits)
    {
        remainder = 1;
    }

    // Main loop for pixel output. In the loop, a host write is done only
    // when the correct number of pixels has accumulated to 32 bits. The
    // data accumulated consists of different colors, changing every 4
    // pixels. This is done to show the data more clearly when displayed
    // on the screen.

    colorcycle = 4;

    colorcount = 0;
    color = LIGHTBLUE;
    maxshifts = 32 / modeinfo.bpp;
    shifts = 0;
    shifter = 0;
    hostdata = 0;
    for (loop = 0; loop < src_line_length; loop++)
    {
        // Pack host data by accumulating 32 bits at a time
        pixel = get_color_code (color);
        pixel = pixel << shifter;
        hostdata = hostdata | pixel;
        shifter = shifter + (unsigned long) (modeinfo.bpp);
        shifts++;
        if (shifts >= maxshifts)
        {
            wait_for_fifo (1);
            regw (HOST_DATA0, hostdata);

            shifter = 0;
            shifts = 0;
            hostdata = 0;
        }

        // Change color every 'colorcycle' pixels
        colorcount++;
        if (colorcount >= colorcycle)
        {
            colorcount = 0;
            color++;
            if (color > WHITE)
            {
                color = LIGHTBLUE;
            }
        }
    }

    // If the total number of bits to be written does not divide evenly by
    // 32, the remaining bits will not be written in the loop. In this case,
    // the remaining bits accumulated during the loop are written through
    // host register below.

    if (remainder == 1)
    {
        wait_for_fifo (1);
        regw (HOST_DATA0, hostdata);
    }
}


/* Main C program */

int main (int argc, char *argv[])
{
    int src_line_length, dst_line_length;
    int srcx, srcy;
    int dstx1, dsty1;
    int dstx2, dsty2;

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // Process the command line arguments to override default resolution
    // and color depth settings.
    process_command_line (argc, argv);

    // set an accelerator mode
    if (open_mode (gmode_res, PITCH_XRES, gclr_depth) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // Check for 24 bpp mode - Lines are not supported in 24 bpp modes
    if (modeinfo.bpp == 24)
    {
        // disable accelerator mode and switch back to VGA text mode
        close_mode ();

        printf ("Lines are not supported in 24 bpp modes.\n");
        return (1);
    }

    // initialize standard engine context
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // Setup line source, destination, and lengths

    // source line (a rectangle with a height of 1) at (srcx, srcy)
    srcx = 0;
    srcy = 0;
    src_line_length = modeinfo.xres / 20;

    // first line starting coordinate at (dstx1, dsty1)
    dstx1 = modeinfo.xres / 40;
    dsty1 = modeinfo.yres / 30;

    // second line starting coordinate at (dstx2, dsty2)
    dstx2 = modeinfo.xres / 20;
    dsty2 = modeinfo.yres / 30;

    // destination lines length
    dst_line_length = modeinfo.xres / 6;


    // Draw patterned lines using a source pattern:
    //
    //    A color source pattern is drawn at (srcx, srcy) and has a length
    //    of src_line_length pixels. Using this color source pattern, two
    //    patterned lines are drawn at (dstx1, dsty1) and (dstx2, dsty2) with
    //    a length of dst_line_length pixels. The first line is drawn by
    //    reading the source pattern from left to right, and the second line
    //    is drawn by reading the source pattern from right to left. For each
    //    direction, when the end of the color source pattern is reached, it
    //    will wrap to the start until the entire patterned line is drawn.
    //

    // Setup color source line (rectangle with height of 1):
    //
    //    Setup a color source pattern from host data. The number of host
    //    data writes will depend on the current pixel depth. Also, the host
    //    data must be packed into 32 bit pieces (i.e. for 8bpp modes, each
    //    host write draws 4 pixels).

    // Setup a rectangle fill with host data
    wait_for_fifo (6);
    regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    regw (DP_SRC, FRGD_SRC_HOST);
    regw (DST_X, srcx);
    regw (DST_Y, srcy);
    regw (DST_HEIGHT, 1);
    regw (DST_WIDTH, src_line_length);

    // Fill source rectangle (height of 1) with packed host data
    host_fill (src_line_length);

    // insure that host transfer is done
    wait_for_idle ();

    // Draw a diagonal line using data from the source. When the src data
    // runs out, the source pointer will wrap and repeat according to
    // the source x direction line bit in SRC_CNTL. Both directions are
    // demonstrated.

    /* ---- Draw line using LEFT TO RIGHT source x direction ---- */

    // set source registers to point to (srcx, srcy)
    wait_for_fifo (13);
    regw (DP_SRC, FRGD_SRC_BLIT);
    regw (SRC_CNTL, SRC_LINE_X_LEFT_TO_RIGHT | SRC_PATTERN_ENABLE);
    regw (SRC_X, srcx);
    regw (SRC_Y, srcy);
    regw (SRC_HEIGHT1, 1);
    regw (SRC_WIDTH1, src_line_length);

    // draw a diagonal line at (dstx1, dsty1) of length dst_line_length
    regw (DST_CNTL, DST_LAST_PEL | DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);
    regw (DST_BRES_ERR, 0);
    regw (DST_BRES_INC, 1);
    regw (DST_BRES_DEC, 0x3FFFF);     // -1
    regw (DST_X, dstx1);
    regw (DST_Y, dsty1);
    regw (DST_BRES_LNTH, dst_line_length);     // src data will repeat


    /* ---- Draw line using RIGHT TO LEFT source x direction ---- */

    // set source registers to point to (srcx + src_line_length - 1, srcy)
    // since the source will be read from right to left
    wait_for_fifo (6);
    regw (SRC_CNTL, SRC_LINE_X_RIGHT_TO_LEFT | SRC_PATTERN_ENABLE);
    regw (SRC_X, srcx + src_line_length - 1);
    regw (SRC_Y, srcy);

    // draw a diagonal line at (dstx2, dsty2) of length dst_line_length
    regw (DST_X, dstx2);
    regw (DST_Y, dsty2);
    regw (DST_BRES_LNTH, dst_line_length);

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

