/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * vtcontrl.c - Chapter 8 sample code                                    *
 *                                                                       *
 * To be used with mach64VT/3D RAGE sample code.                         *
 * This module contains routines to control VT/3D RAGE specific          *
 * functions.                                                            *
 *                                                                       *
 * Copyright (c) 1994-1998 ATI Technologies Inc.  All rights reserved.   *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

//#include <i86.h>
#include "atim64vt.h"
#include "definevt.h"
#include "main.h"


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * start_overlay                                                         *
 *  Function: Initializes VT specific registers to perform overlay       *
 *            operations.                                                *
 *    Inputs: NONE                                                       *
 *   Outputs: NONE                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void start_overlay()
{
    unsigned int ISVTGT = 0, ISREVB = 0;

    //  Obtain asic revision information and handle accordingly.
    ISVTGT = is_vtgt();
    ISREVB = is_revb_or_greater();

    wait_for_idle ();

    // Setup overlay (Enable register block 1)
    regw (BUS_CNTL, regr (BUS_CNTL) | 0x08000000);

    // Set the overlay scaler control and scaling values.
    regw (OVERLAY_SCALE_CNTL, SCALE_ZERO_EXTEND | SCALE_RED_TEMP_6500K |
                              SCALE_HORZ_BLEND | SCALE_VERT_BLEND);

    if (!ISREVB)
    {
        regw (BUF0_PITCH, MODE_INFO.pitch);
    }
    else
    {
        regw (SCALER_BUF0_PITCH, MODE_INFO.pitch);
    }

    // Set scaler dimensions and capture configuration.
    regw (CAPTURE_CONFIG, SCALER_FRAME_READ_MODE_FULL |
                          SCALER_BUF_MODE_SINGLE | SCALER_BUF_NEXT_0);

    if (!ISREVB)
    {
        regw (OVERLAY_KEY_CNTL, OVERLAY_MIX_ALWAYS_V | (OVERLAY_EXCLUSIVE_NORMAL<<28));
    }
    else
    {
        regw (OVERLAY_KEY_CNTL, 0x011);
    }

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * setup_overlay                                                         *
 *  Function: Initializes VT specific registers to perform overlay       *
 *            operations given user defined parameters.                  *
 *    Inputs: pixel_format - bits per pixel definition for a provided    *
 *                           image file in memory.                       *
 *            starting_mem_addr - starting address in graphics memory    *
 *                                of the intended source image.          *
 *   Outputs: NONE                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void setup_overlay(int pixel_format, int starting_mem_addr)
{
    unsigned int ISREVB = 0;

    //  Obtain asic revision information and handle accordingly.
    ISREVB = is_revb_or_greater();

    wait_for_fifo (2);

    // Set input video format.
    regw (VIDEO_FORMAT, pixel_format);

    // Set source buffer for capture.
    if (!ISREVB)
    {
        regw (BUF0_OFFSET, starting_mem_addr);
    }
    else
    {
        regw (SCALER_BUF0_OFFSET, starting_mem_addr);
    }

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * enable_overlay                                                        *
 *  Function: Sets a VT specific register parameter to 'on' in order to  *
 *            enable the VT's overlay capabilities.                      *
 *    Inputs: NONE                                                       *
 *   Outputs: NONE                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void enable_overlay()
{
    unsigned long temp;

    wait_for_idle ();

    // Enable the scaler/overlay.
    temp = regr (OVERLAY_SCALE_CNTL);
    temp |= SCALE_ENABLE;
    regw (OVERLAY_SCALE_CNTL, temp);
    temp |= OVERLAY_ENABLE;
    regw (OVERLAY_SCALE_CNTL, temp);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * disable_overlay                                                       *
 *  Function: Disables the VT's overlay capabilities.                    *
 *    Inputs: NONE                                                       *
 *   Outputs: NONE                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void disable_overlay()
{
    unsigned long temp;

    wait_for_idle ();

    // Disable the scaler/overlay.
    temp = regr (OVERLAY_SCALE_CNTL);
    temp &= ~SCALE_ENABLE;
    regw (OVERLAY_SCALE_CNTL, temp);
    temp &= ~OVERLAY_ENABLE;
    regw (OVERLAY_SCALE_CNTL, temp);

}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * finish_overlay                                                        *
 *  Function: Disables the extended register block (memory page          *
 *            enabled).                                                  *
 *    Inputs: NONE                                                       *
 *   Outputs: NONE                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void finish_overlay()
{

    wait_for_idle ();

    regw (BUS_CNTL, regr (BUS_CNTL) & 0xF7FFFFFF);  // Disable register block 1
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * change_overlay_pitch                                                  *
 *  Function: Changes the bpp characteristics of the overlay window.     *
 *    Inputs: int mode_bpp - bits per pixel of image in overlay window.  *
 *   Outputs: NONE                                                       *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void change_overlay_pitch(int mode_bpp)
{

    //  Obtain asic revision information and handle accordingly.
    unsigned int ISVTGT = 0, ISREVB = 0;

    //  Obtain asic revision information and handle accordingly.
    ISVTGT = is_vtgt();
    ISREVB = is_revb_or_greater();

    wait_for_fifo (1);

    if (!ISREVB)
    {
        regw (BUF0_PITCH, (MODE_INFO.pitch * MODE_INFO.bpp) / mode_bpp);
    }
    else
    {
        regw (SCALER_BUF0_PITCH, (MODE_INFO.pitch * MODE_INFO.bpp) / mode_bpp);
    }

}
